package gov.va.genisis2.dao.impl;

import gov.va.genisis2.dao.ILookUpDao;
import gov.va.genisis2.exceptions.GenisisDAOException;
import gov.va.genisis2.model.CommentType;
import gov.va.genisis2.model.RequestType;
import gov.va.genisis2.model.Source;
import gov.va.genisis2.model.WorkflowStatus;

import java.util.List;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

/**
 * The Class LookUpDao.
 * 
 * The LookUp data access object (DAO) is an object that provides an abstract
 * interface to some type of database or other persistence mechanism. By mapping
 * application calls to the persistence layer, LookUp DAO provide some specific
 * data operations without exposing details of the database.
 */
@Repository
@Transactional(value = "transactionManager")
public class LookUpDao implements ILookUpDao {
	
	/** The LOGGER. */
	private static final Logger LOGGER = LoggerFactory.getLogger(LookUpDao.class);
	
	/** The sessionFactory. */
	@Autowired
	@Qualifier("sessionFactory")
	private SessionFactory sessionFactory;

	/**
	 * This method is used to get status by id.
	 * 
	 * @param statusDescription
	 *            The statusDescription.
	 * @return int This returnes statusId.
	 */
	@Override
	public int getStatusId(String statusDescription) throws GenisisDAOException {
		int statusId;
		List<WorkflowStatus> requestList = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<WorkflowStatus> query = criteriaBuilder.createQuery(WorkflowStatus.class);
			Root<WorkflowStatus> root = query.from(WorkflowStatus.class);
			Predicate condition = criteriaBuilder.equal(root.get("description"), statusDescription);
			query.where(condition);
			requestList = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while doing lookup on " + statusDescription + ". Reason: " + ex.getMessage(), ex);
			return 0;
		}
		if (requestList == null || (requestList.isEmpty())) {
			statusId = 0;
			LOGGER.error(" LookUp is not present in WorkStatus. {}", statusDescription);
		} else {
			statusId = requestList.get(0).getId();
		}
		return statusId;
	}

	/**
	 * This method is used to getRequestType.
	 * 
	 * @param id
	 *            The id.
	 * @return RequestType This returns RequestType.
	 */
	@Override
	public RequestType getRequestType(int id) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getRequestType:  {}",id);
		}
		RequestType requestType = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			requestType = session.get(RequestType.class, id);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying getRequestType.", ex);
			throw new GenisisDAOException("Exception occurred while querying getRequestType.", ex);
		}
		return requestType;
	}

	/**
	 * This method is used to getSource.
	 * 
	 * @param id
	 *            The id.
	 * @return Source This returns Source.
	 */
	@Override
	public Source getSource(int id) throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getSource:  {}",id);
		}

		Source source = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			source = session.get(Source.class, id);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying getSource.", ex);
			throw new GenisisDAOException("Exception occurred while querying getSource.", ex);
		}
		return source;
	}

	/**
	 * This method is used to getDataSources.
	 * 
	 * @return this returns list of Source.
	 */
	@Override
	public List<Source> getDataSources() throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getDataSources: ");
		}

		List<Source> listSource = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<Source> query = criteriaBuilder.createQuery(Source.class);
			Root<Source> root = query.from(Source.class);
			query.select(root);
			listSource = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getDataSources.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getDataSources.", ex);
		}
		return listSource;
	}

	/**
	 * This method is used to getRequestTypes.
	 * 
	 * @return this returns list of RequestType.
	 */
	@Override
	public List<RequestType> getRequestTypes() throws GenisisDAOException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("getRequestTypes: ");
		}

		List<RequestType> listRequestType = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			CriteriaBuilder criteriaBuilder = session.getCriteriaBuilder();
			CriteriaQuery<RequestType> query = criteriaBuilder.createQuery(RequestType.class);
			Root<RequestType> root = query.from(RequestType.class);
			query.select(root);
			listRequestType = session.createQuery(query).getResultList();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying  getRequestTypes.", ex);
			throw new GenisisDAOException("Exception occurred while querying  getRequestTypes.", ex);
		}
		return listRequestType;
	}

	@Override
	public CommentType getCommentType(int id) throws GenisisDAOException {
		LOGGER.info("getCommentType: {}",id);
		CommentType commentType = null;
		try {
			Session session = sessionFactory.getCurrentSession();
			commentType = session.get(CommentType.class, id);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying getCommentType.", ex);
			throw new GenisisDAOException("Exception occurred while querying getCommentType.", ex);
		}
		return commentType;
	}
}
